import Combine
import SwiftUI

struct RandomIdentifiable<T>: Identifiable {
    let id = UUID()
    let value: T
}

struct MinWidthButton: View {
    private let width: CGFloat
    private let title: String
    private let action: () -> Void
    
    init(_ title: String, width: CGFloat = 120, action: @escaping () -> Void) {
        self.width = width
        self.title = title
        self.action = action
    }
    
    var body: some View {
        Button(action: action) {
            Text(title)
                .frame(minWidth: width)
        }
    }
}

typealias KeyValue = (key: String, value: CustomStringConvertible?)
struct KeyValueView: View {
    let pair: KeyValue
    
    var body: some View {
        HStack {
            Text("\(pair.key):")
                .frame(width: 150, alignment: .leading)
            Text(pair.value.flatMap { $0.description } ?? "")
            Spacer(minLength: 0)
        }
        .frame(maxWidth: .infinity)
    }
}

extension View {
    func performAction(name: String, successAlert: Bool = true, body: () throws -> Void) {
        var showAlert = successAlert
        let alert = NSAlert()
        alert.messageText = name
        do {
            try body()
            alert.informativeText = "\(name) succeeds"
            alert.alertStyle = .informational
        } catch {
            alert.informativeText = error.localizedDescription
            alert.alertStyle = .critical
            showAlert = true
        }
        if showAlert {
            alert.runModal()
        }
        ViewUpdaterHack.shared.update()
    }
    
    func showAlert(title: String, text: String, style: NSAlert.Style) {
        let alert = NSAlert()
        
        alert.messageText = title
        alert.informativeText = text
        alert.alertStyle = style
        
        alert.runModal()
    }
    
    func locationForFile(save: Bool) -> URL? {
        let panel: NSSavePanel
        if save {
            panel = NSSavePanel()
        } else {
            let openPanel = NSOpenPanel()
            openPanel.allowsMultipleSelection = false
            openPanel.canChooseFiles = true
            openPanel.canChooseDirectories = true
            panel = openPanel
        }
        _ = panel.runModal()
        return panel.url
    }
    
    /// A backwards compatible wrapper for iOS 14 `onChange`
    @ViewBuilder func valueChanged<T: Equatable>(value: T, onChange: @escaping (T) -> Void) -> some View {
        if #available(macOS 11.0, iOS 14.0, *) {
            self.onChange(of: value, perform: onChange)
        } else {
            self.onReceive(Just(value)) { (value) in
                onChange(value)
            }
        }
    }
}

class ViewUpdaterHack: ObservableObject, @unchecked Sendable {
    @Published var updateCount = 0
    
    static let shared = ViewUpdaterHack()
    
    func update() {
        updateCount += 1
    }
}

extension Date {
    func humanReadable() -> String {
        let formatter = DateFormatter()
        formatter.dateFormat = "E MMM d yyyy HH:mm:ssZ";
        return formatter.string(from: self)
    }
}
